/***************************************************************** 
*  @file CRYS_HOST_TST_RSA.c
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/



/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "CRYS_HOST_TST_RSA.h"
#include "ExtApp_CRYS_Defines.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

 
/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PRIME_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes RSA primitive encryption/decryption process test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PRIME_EncDec (DxUint8_t	*PubExponent_ptr,
									DxUint16_t	 PubExponentSize,
									DxUint8_t   *PrivExponent_ptr,
									DxUint16_t   PrivExponentSize,
									DxUint8_t	*Modulus_ptr,
									DxUint16_t	 KeySize,
									DxUint8_t    CRT_Mode,
									DxUint8_t   *P_ptr, 
									DxUint16_t	 PSize,
									DxUint8_t   *Q_ptr,
									DxUint16_t   QSize,
									DxUint8_t   *dP_ptr, 
									DxUint16_t   dPSize,
									DxUint8_t   *dQ_ptr,
									DxUint16_t   dQSize,
									DxUint8_t   *qInv_ptr,
									DxUint16_t   qInvSize,
									DxUint8_t   *DataIn_ptr,
									DxUint16_t   DataInSize,
									DxUint8_t   *ExpectedData_ptr,
									char        *FailureFunctionNameOut_ptr,
        						    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PRIME_EncDec_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;        
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send public exponent */
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* send ExpectedData_ptr                                    */
   /* The buffer must be at least PubKey_ptr->N.len bytes long */
   /* (that is, the size of the modulus, in bytes).            */
   //Host_maxLength = ((KeySize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr, 
                                     KeySize, 
                                     Host_maxLength,
                                     &Host_sramOffset, 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
 

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PRIME_EncDec */




/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PRIME_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes RSA primitive encryption/decryption process test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PRIME_SignVerify (DxUint8_t	*PubExponent_ptr,
									DxUint16_t	 PubExponentSize,
									DxUint8_t   *PrivExponent_ptr,
									DxUint16_t   PrivExponentSize,
									DxUint8_t	*Modulus_ptr,
									DxUint16_t	 KeySize,
									DxUint8_t    CRT_Mode,
									DxUint8_t   *P_ptr, 
									DxUint16_t	 PSize,
									DxUint8_t   *Q_ptr,
									DxUint16_t   QSize,
									DxUint8_t   *dP_ptr, 
									DxUint16_t   dPSize,
									DxUint8_t   *dQ_ptr,
									DxUint16_t   dQSize,
									DxUint8_t   *qInv_ptr,
									DxUint16_t   qInvSize,
									DxUint8_t   *DataIn_ptr,
									DxUint16_t   DataInSize,
									DxUint8_t   *ExpectedData_ptr,
									char        *FailureFunctionNameOut_ptr,
        						    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PRIME_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;        
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send public exponent */
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* send ExpectedData_ptr                                    */
   /* The buffer must be at least PubKey_ptr->N.len bytes long */
   /* (that is, the size of the modulus, in bytes).            */
   //Host_maxLength = ((KeySize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr, 
                                     KeySize, 
                                     Host_maxLength,
                                     &Host_sramOffset, 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
 

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PRIME_SignVerify */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* CRT_Mode - mode of the private key
* KeyGenType - type how to generate key (internal or build from external key)
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes RSA PKCS1v15 encrypt and decrypt according to PKCS#1 Ver1.5 test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PKCS1v15_EncDec (DxUint8_t	*PubExponent_ptr,
									   DxUint16_t	 PubExponentSize,
									   DxUint8_t   *PrivExponent_ptr,
									   DxUint16_t   PrivExponentSize,
									   DxUint8_t	*Modulus_ptr,
									   DxUint16_t	 KeySize,
									   DxUint16_t   KeyGenType,
									   DxUint8_t    CRT_Mode,
									   DxUint8_t   *P_ptr, 
									   DxUint16_t	 PSize,
									   DxUint8_t   *Q_ptr,
									   DxUint16_t   QSize,
									   DxUint8_t   *dP_ptr, 
									   DxUint16_t   dPSize,
									   DxUint8_t   *dQ_ptr,
									   DxUint16_t   dQSize,
									   DxUint8_t   *qInv_ptr,
									   DxUint16_t   qInvSize,
									   DxUint8_t   *DataIn_ptr,
									   DxUint16_t   DataInSize,
									   char        *FailureFunctionNameOut_ptr,
									   DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PKCS1v15_EncDec_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;  
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

 
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PKCS1v15_EncDec */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify (
                                DxUint8_t	*PubExponent_ptr,
								DxUint16_t	 PubExponentSize,
								DxUint8_t   *PrivExponent_ptr,
								DxUint16_t   PrivExponentSize,
								DxUint8_t	*Modulus_ptr,
								DxUint16_t	 KeySize,
								DxUint8_t    CRT_Mode,
								DxUint8_t    KeyGenType,
								CRYS_RSA_HASH_OpMode_t hashMode,
								DxUint8_t   *P_ptr, 
								DxUint16_t	 PSize,
								DxUint8_t   *Q_ptr,
								DxUint16_t   QSize,
								DxUint8_t   *dP_ptr, 
								DxUint16_t   dPSize,
								DxUint8_t   *dQ_ptr,
								DxUint16_t   dQSize,
								DxUint8_t   *qInv_ptr,
								DxUint16_t   qInvSize,
								DxUint8_t   *DataIn_ptr,
								DxUint16_t   DataInSize,
								char        *FailureFunctionNameOut_ptr,
							    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PKCS1v15_NonIntegrate_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = hashMode;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

 
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PKCS1v15_NonIntegrate_SignVerify */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_Integrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PKCS1v15_Integrate_SignVerify (
                                DxUint8_t	*PubExponent_ptr,
								DxUint16_t	 PubExponentSize,
								DxUint8_t   *PrivExponent_ptr,
								DxUint16_t   PrivExponentSize,
								DxUint8_t	*Modulus_ptr,
								DxUint16_t	 KeySize,
								DxUint8_t    CRT_Mode,
								DxUint8_t    KeyGenType,
								CRYS_RSA_HASH_OpMode_t hashMode,
								DxUint8_t   *P_ptr, 
								DxUint16_t	 PSize,
								DxUint8_t   *Q_ptr,
								DxUint16_t   QSize,
								DxUint8_t   *dP_ptr, 
								DxUint16_t   dPSize,
								DxUint8_t   *dQ_ptr,
								DxUint16_t   dQSize,
								DxUint8_t   *qInv_ptr,
								DxUint16_t   qInvSize,
								DxUint8_t   *DataIn_ptr,
								DxUint16_t   DataInSize,
								char        *FailureFunctionNameOut_ptr,
							    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PKCS1v15_Integrate_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = hashMode;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
     
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PKCS1v15_Integrate_SignVerify */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_OAEP_EncDec         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_OAEP_EncDec (
                                DxUint8_t	*PubExponent_ptr,
                    			DxUint16_t	 PubExponentSize,
                    			DxUint8_t   *PrivExponent_ptr,
                    			DxUint16_t   PrivExponentSize,
                    			DxUint8_t	*Modulus_ptr,
                    			DxUint16_t	 KeySize,
                    			DxUint8_t    CRT_Mode,
                    			DxUint8_t    KeyGenType,
                    			CRYS_RSA_HASH_OpMode_t hashMode,
                    			DxUint8_t   *P_ptr, 
                    			DxUint16_t	 PSize,
                    			DxUint8_t   *Q_ptr,
                    			DxUint16_t   QSize,
                    			DxUint8_t   *dP_ptr, 
                    			DxUint16_t   dPSize,
                    			DxUint8_t   *dQ_ptr,
                    			DxUint16_t   dQSize,
                    			DxUint8_t   *qInv_ptr,
                    			DxUint16_t   qInvSize,
                    			DxUint8_t   *DataIn_ptr,
                    			DxUint16_t   DataInSize,
                    			char        *FailureFunctionNameOut_ptr,
							    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_OAEP_EncDec_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = hashMode;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_OAEP_EncDec */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_NonIntegrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* SaltLen - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
   						Typical lengths are 0 and hLen (20 for SHA1)
   						The maximum length allowed is NSize - hLen - 2
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes test for RSA PSS PKCS#1v2.1 non integrated sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PSS_NonIntegrate_SignVerify (
                                  DxUint8_t	            *PubExponent_ptr,
								  DxUint16_t	         PubExponentSize,
								  DxUint8_t             *PrivExponent_ptr,
								  DxUint16_t             PrivExponentSize,
								  DxUint8_t	            *Modulus_ptr,
								  DxUint16_t	         KeySize,
								  DxUint8_t              CRT_Mode,
								  DxUint8_t				 KeyGenType,
								  CRYS_RSA_HASH_OpMode_t HashMode,
								  DxUint16_t		     SaltLength,
								  DxUint8_t             *P_ptr, 
								  DxUint16_t	         PSize,
								  DxUint8_t             *Q_ptr,
								  DxUint16_t             QSize,
								  DxUint8_t             *dP_ptr, 
								  DxUint16_t             dPSize,
								  DxUint8_t             *dQ_ptr,
								  DxUint16_t             dQSize,
								  DxUint8_t             *qInv_ptr,
								  DxUint16_t             qInvSize,
								  DxUint8_t             *DataIn_ptr,
								  DxUint16_t             DataInSize,
								  char                  *FailureFunctionNameOut_ptr,
							      DxUint32_t             FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PSS_NonIntegrate_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = HashMode;
    Host_messageParam[DX_SEP_HOST_RSA_SALTLENGTH_OFFSET] = SaltLength;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PSS_NonIntegrate_SignVerify */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_Integrate_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* SaltLen - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
   						Typical lengths are 0 and hLen (20 for SHA1)
   						The maximum length allowed is NSize - hLen - 2
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes test for RSA PSS PKCS#1v2.1 integrated sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PSS_Integrate_SignVerify (
                                  DxUint8_t	            *PubExponent_ptr,
								  DxUint16_t	         PubExponentSize,
								  DxUint8_t             *PrivExponent_ptr,
								  DxUint16_t             PrivExponentSize,
								  DxUint8_t	            *Modulus_ptr,
								  DxUint16_t	         KeySize,
								  DxUint8_t              CRT_Mode,
								  DxUint8_t				 KeyGenType,
								  CRYS_RSA_HASH_OpMode_t HashMode,
								  DxUint16_t		     SaltLength,
								  DxUint8_t             *P_ptr, 
								  DxUint16_t	         PSize,
								  DxUint8_t             *Q_ptr,
								  DxUint16_t             QSize,
								  DxUint8_t             *dP_ptr, 
								  DxUint16_t             dPSize,
								  DxUint8_t             *dQ_ptr,
								  DxUint16_t             dQSize,
								  DxUint8_t             *qInv_ptr,
								  DxUint16_t             qInvSize,
								  DxUint8_t             *DataIn_ptr,
								  DxUint16_t             DataInSize,
								  char                  *FailureFunctionNameOut_ptr,
							      DxUint32_t             FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PSS_Integrate_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = HashMode;
    Host_messageParam[DX_SEP_HOST_RSA_SALTLENGTH_OFFSET] = SaltLength;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PSS_Integrate_SignVerify */





/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PKCS1v15_ExternalHash_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
*  dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes non integrated test for RSA PKCS1v15 Sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PKCS1v15_ExternalHash_SignVerify (
                                DxUint8_t	*PubExponent_ptr,
								DxUint16_t	 PubExponentSize,
								DxUint8_t   *PrivExponent_ptr,
								DxUint16_t   PrivExponentSize,
								DxUint8_t	*Modulus_ptr,
								DxUint16_t	 KeySize,
								DxUint8_t    CRT_Mode,
								DxUint8_t    KeyGenType,
								CRYS_RSA_HASH_OpMode_t hashMode,
								DxUint8_t   *P_ptr, 
								DxUint16_t	 PSize,
								DxUint8_t   *Q_ptr,
								DxUint16_t   QSize,
								DxUint8_t   *dP_ptr, 
								DxUint16_t   dPSize,
								DxUint8_t   *dQ_ptr,
								DxUint16_t   dQSize,
								DxUint8_t   *qInv_ptr,
								DxUint16_t   qInvSize,
								DxUint8_t   *DataIn_ptr,
								DxUint16_t   DataInSize,
								char        *FailureFunctionNameOut_ptr,
							    DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PKCS1v15_ExternalHash_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = hashMode;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
     
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PKCS1v15_Integrate_SignVerify */




/***********************************************************************
* Function Name:                                                           
*  SEPQA_RSA_PSS_ExternalHash_SignVerify         
* 
* Inputs:
* PubExponent_ptr - a pointer to the exponent stream of bytes ( Big endian ).
* PubExponentSize - The size of the exponent in bytes.  
* Modulus_ptr  - a pointer to the modulus stream of bytes ( Big endian ) the MS bit must be set to '1'.
* KeySize (ModulusSize)  - The size of the modulus in bytes.
* KeyGenType - key gen type (internal/external)
* CRT_Mode - mode of the private key
* hashMode - hash mode
* SaltLen - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
   						Typical lengths are 0 and hLen (20 for SHA1)
   						The maximum length allowed is NSize - hLen - 2
* P_ptr - A pointer to the first factor stream of bytes (Big-Endian format)
* PSize - The size of the first factor, in bytes.
* Q_ptr - A pointer to the second factor stream of bytes (Big-Endian format)
* QSize - The size of the second factor, in bytes.
* dP_ptr - A pointer to the first factor's CRT exponent stream of bytes (Big-Endian format)
* dPSize - The size of the first factor's CRT exponent, in bytes.
* dQ_ptr - A pointer to the second factor's CRT exponent stream of bytes (Big-Endian format)
* dQSize - The size of the second factor's CRT exponent, in bytes.
* qInv_ptr - A pointer to the first CRT coefficient stream of bytes (Big-Endian format)
* qInvSize - The size of the first CRT coefficient, in bytes.
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes test for RSA PSS PKCS#1v2.1 integrated sign & Verify
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_RSA_PSS_ExternalHash_SignVerify (
                                  DxUint8_t	            *PubExponent_ptr,
								  DxUint16_t	         PubExponentSize,
								  DxUint8_t             *PrivExponent_ptr,
								  DxUint16_t             PrivExponentSize,
								  DxUint8_t	            *Modulus_ptr,
								  DxUint16_t	         KeySize,
								  DxUint8_t              CRT_Mode,
								  DxUint8_t				 KeyGenType,
								  CRYS_RSA_HASH_OpMode_t HashMode,
								  DxUint16_t		     SaltLength,
								  DxUint8_t             *P_ptr, 
								  DxUint16_t	         PSize,
								  DxUint8_t             *Q_ptr,
								  DxUint16_t             QSize,
								  DxUint8_t             *dP_ptr, 
								  DxUint16_t             dPSize,
								  DxUint8_t             *dQ_ptr,
								  DxUint16_t             dQSize,
								  DxUint8_t             *qInv_ptr,
								  DxUint16_t             qInvSize,
								  DxUint8_t             *DataIn_ptr,
								  DxUint16_t             DataInSize,
								  char                  *FailureFunctionNameOut_ptr,
							      DxUint32_t             FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_PSS_ExternalHash_SignVerify_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = DataInSize;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = KeyGenType;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = HashMode;
    Host_messageParam[DX_SEP_HOST_RSA_SALTLENGTH_OFFSET] = SaltLength;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   /* send public exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
      
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send DataIn_ptr */
   Host_maxLength = ((DataInSize) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_RSA_PSS_Integrate_SignVerify */


EXPORT_C DxUint32  SEPQA_RSA_Get_PrivAndPubKey (DxUint8_t	*PubExponent_ptr,
									        	DxUint16_t	 PubExponentSize,
												DxUint8_t   *PrivExponent_ptr,
												DxUint16_t   PrivExponentSize,
												DxUint8_t	*Modulus_ptr,
												DxUint16_t	 KeySize,
												DxUint8_t    CRT_Mode,
												DxUint8_t   *P_ptr, 
												DxUint16_t	 PSize,
												DxUint8_t   *Q_ptr,
												DxUint16_t   QSize,
												DxUint8_t   *dP_ptr, 
												DxUint16_t   dPSize,
												DxUint8_t   *dQ_ptr,
												DxUint16_t   dQSize,
												DxUint8_t   *qInv_ptr,
												DxUint16_t   qInvSize,
												char        *FailureFunctionNameOut,
												DxUint32_t   FuncNameMaxSize)
{
	  /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_RSA_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_Get_PrivAndPubKey_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
     Host_messageParam[DX_SEP_HOST_RSA_DATASIZE_IN_OFFSET] = 0;
    Host_messageParam[DX_SEP_HOST_RSA_PUBEXPONENTSIZE_OFFSET] = PubExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_PRIVEXPONENTSIZE_OFFSET] = PrivExponentSize;
    Host_messageParam[DX_SEP_HOST_RSA_KEYSIZE_OFFSET] = KeySize;        
    Host_messageParam[DX_SEP_HOST_RSA_CRTMODE_OFFSET] = CRT_Mode;
    Host_messageParam[DX_SEP_HOST_RSA_KEYGENTYPE_OFFSET] = 0;
    Host_messageParam[DX_SEP_HOST_RSA_HASHMODE_OFFSET] = 0;
    Host_messageParam[DX_SEP_HOST_RSA_SALTLENGTH_OFFSET] = 0;
    Host_messageParam[DX_SEP_HOST_RSA_PSIZE_OFFSET] = PSize;        
    Host_messageParam[DX_SEP_HOST_RSA_QSIZE_OFFSET] = QSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dPSIZE_OFFSET] = dPSize;        
	Host_messageParam[DX_SEP_HOST_RSA_dQSIZE_OFFSET] = dQSize;        
	Host_messageParam[DX_SEP_HOST_RSA_qInvSIZE_OFFSET] = qInvSize;        
	       
	
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send public exponent */
   Host_maxLength = CRYS_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PubExponent_ptr , 
                                     PubExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send private exponent */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)PrivExponent_ptr , 
                                     PrivExponentSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send modulus */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Modulus_ptr , 
                                     KeySize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send P_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)P_ptr , 
                                     PSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dP_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dP_ptr , 
                                     dPSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send dQ_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)dQ_ptr , 
                                     dQSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send qInv_ptr */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)qInv_ptr , 
                                     qInvSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                                                      
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
 

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
}
